
import torch
import torch.nn as nn
import numpy as np
import joblib
import cma
import pandas as pd
import os

# ✅ Matching Sequential decoder
latent_dim = 8
output_dim = 200

decoder = nn.Sequential(
    nn.Linear(latent_dim, 128),
    nn.ReLU(),
    nn.Linear(128, output_dim)
)
decoder.load_state_dict(torch.load("decoder.pt"))
decoder.eval()

# ✅ Surrogate model
class SurrogateMLP(nn.Module):
    def __init__(self, input_dim, output_dim):
        super(SurrogateMLP, self).__init__()
        self.net = nn.Sequential(
            nn.Linear(input_dim, 256),
            nn.ReLU(),
            nn.Linear(256, 256),
            nn.ReLU(),
            nn.Linear(256, output_dim)
        )
    def forward(self, x):
        return self.net(x)

geometry_dim = output_dim
input_dim = geometry_dim + 1
surrogate = SurrogateMLP(input_dim=input_dim, output_dim=3)
surrogate.load_state_dict(torch.load("surrogate_model.pt"))
surrogate.eval()

# ✅ Load scalers
scaler_X = joblib.load("X_scaler.pkl")
scaler_Y = joblib.load("Y_scaler.pkl")

# ✅ Robust objective with basic constraints
def objective(z):
    z_tensor = torch.tensor(z, dtype=torch.float32).unsqueeze(0)
    geometry = decoder(z_tensor).detach().numpy().flatten()

    # Split
    N_POINTS = len(geometry) // 2
    yu = geometry[:N_POINTS]
    yl = geometry[N_POINTS:]

    # Thickness check
    min_thickness = np.min(yu - yl)
    if min_thickness < 0.02:
        print("Rejected: Too thin")
        return 1e6

    # Smoothness check (spikes)
    dy_upper = np.diff(yu)
    dy_lower = np.diff(yl)
    max_slope = max(np.max(np.abs(dy_upper)), np.max(np.abs(dy_lower)))
    if max_slope > 0.5:
        print("Rejected: Spiky surface")
        return 1e6

    # Surrogate prediction
    aoa = 4.0
    X_input = np.concatenate([geometry, [aoa]]).reshape(1, -1)
    X_scaled = scaler_X.transform(X_input)

    X_tensor = torch.tensor(X_scaled, dtype=torch.float32)
    with torch.no_grad():
        Y_scaled = surrogate(X_tensor).numpy()
    Y = scaler_Y.inverse_transform(Y_scaled)
    Cl, Cd = Y[0][0], Y[0][1]
    score = - (Cl / Cd)

    print(f"Accepted: Cl={Cl:.3f}, Cd={Cd:.4f}, Cl/Cd={Cl/Cd:.1f}")
    return score

# ✅ Run CMA-ES
es = cma.CMAEvolutionStrategy(latent_dim * [0], 0.5)
es.optimize(objective, iterations=30)

# ✅ Save result
best_z = es.result.xbest
z_tensor = torch.tensor(best_z, dtype=torch.float32).unsqueeze(0)
best_geometry = decoder(z_tensor).detach().numpy().flatten()

os.makedirs("new_airfoils", exist_ok=True)
np.savetxt("new_airfoils/new_airfoil_shape.csv", best_geometry)
print("✅ Saved new airfoil geometry to new_airfoils/new_airfoil_shape.csv")
