import pandas as pd
import numpy as np
import os

# Config
AIRFOIL_DIR = "airfoils_uiuc"
N_POINTS = 100  # Number of chordwise samples per surface

# Prepare storage for geometry data
rows = []

for filename in os.listdir(AIRFOIL_DIR):
    if not filename.lower().endswith(".dat"):
        continue
    airfoil_name = os.path.splitext(filename)[0]
    coords = np.loadtxt(os.path.join(AIRFOIL_DIR, filename), skiprows=1)

    # Find index of leading edge (minimum x)
    idx_le = np.argmin(coords[:, 0])

    # Separate upper (TE->LE) and lower (LE->TE) surfaces
    upper = coords[:idx_le + 1][::-1]  # reverse to LE->TE
    lower = coords[idx_le:]

    # Uniform chordwise x positions
    xs = np.linspace(0.0, 1.0, N_POINTS)

    # Interpolate y-coordinates on upper and lower surfaces
    y_upper = np.interp(xs, upper[:, 0], upper[:, 1])
    y_lower = np.interp(xs, lower[:, 0], lower[:, 1])

    # Append row: airfoil name + y_upper + y_lower
    rows.append([airfoil_name] + list(y_upper) + list(y_lower))

# Build DataFrame
columns = ["Airfoil"] + [f"yu_{i}" for i in range(N_POINTS)] + [f"yl_{i}" for i in range(N_POINTS)]
geometry_df = pd.DataFrame(rows, columns=columns)

# Save to CSV
output_path = "geometry.csv"
geometry_df.to_csv(output_path, index=False)

print(f"Generated {output_path} with shape {geometry_df.shape}")
